﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Diagnostics;
using System.Threading.Tasks;
using VeteransAffairs.Registries.BusinessAHOBPR.MilitaryInfoService;
using VeteransAffairs.Registries.BusinessAHOBPR;

namespace VeteransAffairs.Registries.BusinessManagerAHOBPR.Emis
{
    public class MilitaryServiceEpisodeManager : AHOBPRBaseBO, IUpdateManager
    {
        const string FUTUREDATE = "9999-11-11";
        string connectionInfo = string.Empty;

        public MilitaryServiceEpisodeManager() { }
        public MilitaryServiceEpisodeManager(string dbConnectionInfo)
        {
            connectionInfo = dbConnectionInfo;
        }
        public string Name
        {
            get
            {
                return "Service Episodes";
            }
        }
        public bool Update(string edipi, int registrantId)
        {
            var successfullyProcessed = true;
            try
            {
                var importer = new MilitaryServiceEpisodeImporter();
                List<MilitaryServiceEpisode> emisEpisodes = importer.GetMilitaryServiceEpisodes(edipi);
                if (emisEpisodes != null)
                {
                    using (var db = GetLocalContext(connectionInfo))
                    {
                        List<REGISTRANT_SERVICE_EPISODE> serviceEpisodes = CreateDBObjectsFromEmisData(emisEpisodes, registrantId);
                        successfullyProcessed = ProcessServiceEpisodes(serviceEpisodes, registrantId, db);
                        if (successfullyProcessed)
                        {
                            var registrant = db.REGISTRANTs.FirstOrDefault(r => r.REGISTRANT_ID == registrantId);
                            if (registrant != null)
                            {
                                registrant.SetAsChangeTrackingRoot();
                                registrant.SetAsUpdateOnSubmit();
                                UpdateRegistrantComponentIdAndDischargeCharacterId(emisEpisodes, registrant, db);
                                db.SubmitChanges();
                            }
                            else
                            {
                                successfullyProcessed = false;
                                AHOBPRLogger.LogErrorMessage("Update Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, $"No registrant entry in DB, when searching for registrant with ID of: {registrantId}");
                            }
                        }
                    }
                }
                //update registrant object with veteranstatus information
            }
            catch (Exception ex)
            {
                var errormessage = "MilitaryServiceEpisodeManager.Update(): " + ex.Message;
                Trace.WriteLine(errormessage);
                AHOBPRLogger.LogErrorMessage("eMIS Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                successfullyProcessed = false;
            }

            return successfullyProcessed;
        }
        public bool Update(string edipi)
        {
            var result = false;
            try
            {
                using (_dbAhobpr = GetDataContext())
                {
                    var registrant = _dbAhobpr.REGISTRANTs.FirstOrDefault(e => e.EDIPI == edipi);
                    if (registrant != null)
                    {
                        result = Update(edipi, registrant.REGISTRANT_ID);
                    }
                }
            }
            catch (Exception ex)
            {
                var errormessage = "MilitaryServiceEpisodeManager.Update(): " + ex.Message;
                Trace.WriteLine(errormessage);
                AHOBPRLogger.LogErrorMessage("Exception", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
            }
            return result;
        }

        public bool ProcessServiceEpisodes(List<REGISTRANT_SERVICE_EPISODE> emisEpisodes, int registrantId, AHOBPRDataAccess dbAhobpr)
        {
            var successfullyProcessed = true;
            var okToSubmit = false;
            try
            {
                var newServiceEpisodes = new List<REGISTRANT_SERVICE_EPISODE>();
                var dbEntries = dbAhobpr.REGISTRANT_SERVICE_EPISODEs
                     .Where(e => e.REGISTRANT_ID == registrantId)
                     .ToList();
                foreach (var episode in emisEpisodes)
                {
                    var match = dbEntries.FirstOrDefault(e => e.HasSameStartAndEndDate(episode));

                    if (match != null)
                    {
                        match.SetAsChangeTrackingRoot();
                        //Update if there is a difference
                        if (!match.Equals(episode))
                        {
                            okToSubmit = true;
                            SetBranchOfService(match, episode.STD_BRANCH_OF_SERVICE_ID, dbAhobpr);
                            match.Update(episode);
                        }
                    }
                    else
                    {
                        //Add to registrant_service_episodes
                        newServiceEpisodes.Add(episode);
                    }
                }
                if (newServiceEpisodes.Count > 0)
                {
                    dbAhobpr.REGISTRANT_SERVICE_EPISODEs.InsertAllOnSubmit(newServiceEpisodes);
                    okToSubmit = true;
                }
                if (okToSubmit)
                {
                    dbAhobpr.SubmitChanges();
                }
            }
            catch (Exception ex)
            {
                var errormessage = "MilitaryServiceEpisodeManager.ProcessServiceEpisodes: Exception found: " + ex.Message;
                Trace.WriteLine(errormessage);
                AHOBPRLogger.LogErrorMessage("Error", this.GetType().Name + "." + System.Reflection.MethodBase.GetCurrentMethod().Name, errormessage);
                successfullyProcessed = false;
            }
            return successfullyProcessed;
        }
        private static void UpdateRegistrantDischargeCharacterId(REGISTRANT registrant, MilitaryServiceEpisode se, AHOBPRDataAccess db)
        {
            if (!string.IsNullOrEmpty(se?.militaryServiceEpisodeData?.dischargeCharacterOfServiceCode))
            {
                registrant.STD_DISCHARGE_CHARACTER_ID = new AHOBPRGenericManager().GetStdDischargeCharacterId(se.militaryServiceEpisodeData.dischargeCharacterOfServiceCode);
                registrant.STD_DISCHARGE_CHARACTER = db.STD_DISCHARGE_CHARACTERs.FirstOrDefault(d => d.STD_DISCHARGE_CHARACTER_ID == registrant.STD_DISCHARGE_CHARACTER_ID);
            }
        }
        private static void UpdateRegistrantComponentId(REGISTRANT registrant, MilitaryServiceEpisode se)
        {
            if (!string.IsNullOrEmpty(se?.keyData?.personnelCategoryTypeCode))
            {
                registrant.STD_COMPONENT_ID_CURRENT = AHOBPRShared.GetStdComponent(se.keyData.personnelCategoryTypeCode);
            }
        }

        private void UpdateRegistrantComponentIdAndDischargeCharacterId(List<MilitaryServiceEpisode> serviceEpisodes, REGISTRANT registrant, AHOBPRDataAccess db)
        {
            var se = serviceEpisodes.OrderByDescending(s => s.militaryServiceEpisodeData.serviceEpisodeEndDate).FirstOrDefault();
            //Set the registrant's current component ID:
            UpdateRegistrantComponentId(registrant, se);
            UpdateRegistrantDischargeCharacterId(registrant, se, db);
        }
        public Dictionary<DateRange, string> GetPersonnelCategoryCodes(string edipi)
        {
            var result = new Dictionary<DateRange, string>();
            var importer = new MilitaryServiceEpisodeImporter();
            var whatever = importer.GetMilitaryServiceEpisodes(edipi);
            foreach (var item in whatever)
            {
                var range = CreateValidDateRange(item.militaryServiceEpisodeData.serviceEpisodeStartDate, item.militaryServiceEpisodeData.serviceEpisodeEndDate);
                if (!result.Keys.Contains(range)) { result.Add(range, item.keyData.personnelCategoryTypeCode); }
            }
            return result;
        }
        public REGISTRANT_SERVICE_EPISODE CreateREGISTRANT_SERVICE_EPISODE(MilitaryServiceEpisodeData emisInfo, int personnelCategoryTypeCode, int registrantId)
        {
            var serviceEpisode = new REGISTRANT_SERVICE_EPISODE();
            serviceEpisode.REGISTRANT_ID = registrantId;
            serviceEpisode.STD_BRANCH_OF_SERVICE_ID = AHOBPRShared.GetStdBranchOfServiceId(emisInfo.branchOfServiceCode);
            serviceEpisode.BRANCH_OF_SERVICE_TEXT = emisInfo.branchOfServiceCode;
            serviceEpisode.NARRATIVE_REASON_FOR_SEPARATION_TEXT = emisInfo.narrativeReasonForSeparationTxt;
            serviceEpisode.PERSONNEL_CATEGORY_TYPE_CODE = personnelCategoryTypeCode;
            serviceEpisode.SERVICE_EPISODE_END_DATE = (emisInfo.serviceEpisodeEndDate != DateTime.MinValue) ? emisInfo.serviceEpisodeEndDate : (DateTime?)null;
            serviceEpisode.SERVICE_EPISODE_START_DATE = emisInfo.serviceEpisodeStartDate;
            serviceEpisode.SERVICE_EPISODE_TERMINATION_REASON = emisInfo.serviceEpisodeTerminationReason;
            return serviceEpisode;
        }

        private void SetBranchOfService(REGISTRANT_SERVICE_EPISODE serviceEpisode, int? newId, AHOBPRDataAccess db)
        {
            if (serviceEpisode != null && newId != null)
            {
                serviceEpisode.STD_BRANCH_OF_SERVICE = db.STD_BRANCH_OF_SERVICEs.FirstOrDefault(se => se.STD_BRANCH_OF_SERVICE_ID == newId);
            }
        }

        public static DateRange CreateValidDateRange(DateTime startDate, DateTime endDate)
        {
            //If they're active, it'll show up with a DateTime of 0/0/0001
            if (endDate.Day == 1 && endDate.Month == 1 && endDate.Year == 1)
            {
                endDate = DateTime.Parse(FUTUREDATE);
            }
            return new DateRange(startDate, endDate);
        }
        private List<REGISTRANT_SERVICE_EPISODE> CreateDBObjectsFromEmisData(List<MilitaryServiceEpisode> emisData, int registrantId)
        {
            var registrantServiceEpisodes = new List<REGISTRANT_SERVICE_EPISODE>();
            foreach (var item in emisData)
            {
                var personalCategoryTypeCode = AHOBPRShared.GetStdComponent(item.keyData.personnelCategoryTypeCode);
                var episode = CreateREGISTRANT_SERVICE_EPISODE(item.militaryServiceEpisodeData, personalCategoryTypeCode, registrantId);
                registrantServiceEpisodes.Add(episode);
            }
            return registrantServiceEpisodes;
        }

    }

}
